--============================================================
-- testing functions

require "auxiliar"

local name = ...

module(..., package.seeall)

---- datatest = {}

--[[
   Returns a function that checks if a numerical value is in a range.
   x1 and x2 are numerical values 
--]]

---- function datatest.numrange(x1, x2)
function numrange(x1, x2)
  return function (x)
    x1, x2 = math.min(x1, x2), math.max(x1, x2)
    if type(x) ~= "number" or type(x1) ~= "number"
        or type(x2) ~= "number" then
      return nil, "Type mismatch in " .. name .. ".numrange\n"
    end

    if x >= x1 and x <= x2 then
      return true, ""
    else
      return false, "Must be in [" .. x1 .. ", " .. x2 .. "]\n"
    end
  end
end

--[[
   Returns a function that checks if a value is in a set of values.
   t is a homogeneous table with values (strings, numbers, etc.)
   f is a function to transform the target (s) and the fields of t
   before the comparison (v.g., string.upper, math.abs, etc. --it depends)
--]]
---- function datatest.inset(t, f)
function inset(t, f)
  return function (s)
    f = f or function (s) return s end
    local ty = type(t)
    if ty ~= "table" then
      -- unique value to compare
      if ty ~= type(s) then
        return nil, "Type mismatch in ".. name .. ".inset\n"
      end
      if f(t) == f(s) then return true end
      return false, "Must be: " .. tostring(t) .. "\n"
    else
      -- a table of values
      local ty
      local notfirst = false
      for _, v in pairs(t) do
        if notfirst and ty ~= type(v) then
          return nil, "Several types in table in " .. name .. "inset\n"
        end
        ty = type(v) 
        notfirst = true
      end
      if type(s) ~= ty then
        return nil, "Type mismatch in " .. name .. ".inset\n"
      end
      for _, v in pairs(t) do
        if f(v) == f(s) then return true, "" end
      end
      local st, delim = "", " "
      if ty == "string" then delim = ' "' end
      for _, v in pairs(t) do st = st .. delim .. tostring(v) .. delim end
      return false, "Must be one of: " .. st .. "\n"
    end
  end
end

--[[
   Returns a function that checks if a value is a list of elements
   similar to e, containing between n1 and n2 (included) of them.
   e must be a table without subtables (depth 1)
--]]
---- function datatest.listof(e, n1, n2)
function listof(e, n1, n2)
  -- counts the number of elements in a list
  local function numelem(list)
    local n = 0
    for _ in pairs(list) do n = n+1 end
    return n
  end

  -- a function to check element by element the type
  local function equalstruct(e, o)
    for k, v in pairs(e) do
      local w = o[k]
      if type(v) ~= type(w) then return false end
    end
    for k, v in pairs(o) do
      local w = e[k]
      if type(v) ~= type(w) then return false end
    end
    return true
  end
      
  return function (list)
    local n1 = n1 or 1
    local n2 = n2 or math.huge
    local ty = type(e)

    local tot = numelem(list)
    if tot < n1 or tot > n2 then
      return false, "Incorrect number of elements in " .. name .. ".listof [" .. 
                    tostring(n1) .. ", " .. tostring(n2) .. "]\n"
    end

    if ty ~= "table" then
      -- scalar values to compare
      for _, v in pairs(list) do  -- possible negative initial indices
        if type(v) ~= ty then
          return nil, "Type mismatch in " .. name .. ".listof\n"
        end
      end
    else
      local e1
      for _, e in pairs(list) do e1 = e; break end -- stores first elem
      for k, e in pairs(list) do 
        ty = type(e)
        if ty ~= "table" then
          return false, "Not a table in element " ..
                        tonumber(k) .. " in " .. name .. ".listof\n"
        end
        if numelem(e) ~= numelem(e1) then
          return false, 
                 "Different structure in the list in " .. name .. ".listof\n"
        end
        if ty ~= type (e1) then
          return false, "Type mismatch in list in element " ..
                        tonumber(k) .. " in " .. name .. ".listof\n"
        end
      end
      -- a table of values
      for _, v in pairs(list) do
        local ok = equalstruct(e, v)
        if not ok then
          return nil, "Type mismatch in " .. name .. ".listof\n"
        end
      end
      return true, ""
    end
  end
end

----> Other similar datatest.* routines can be developed
----> to check other types of entries

--[[
   Checks if table t agrees with template
   label is only for labelling the table in errors
   The function returns
      true and "" if all is right
      false and an error message in case of mismatch
--]]
---- function datatest.main (template, t, label)  -- the wrapper
function main (template, t, label)  -- the wrapper
  label = label or ""
  local cart, precart, torig = "", "", t

  local function maintest (template, t, allstrict, field)
    field = field or ""
    local ok, contains, val = true, template.CONTAINS, template.VALUE
    local optional, default = template.OPTIONAL, template.DEFAULT
    local strict, treestr = template.STRICT, template.ALLSTRICT
    local test = template.TEST or function (s) return true end
    local lab = "field <" .. label ..
                ((field ~= "") and ("." .. field) or "") .. ">"

    if contains == nil and val == nil then
      cart = cart .. "*1> Error in template: " .. lab .. 
                     " must have fields CONTAINS or VALUE\n"
      return false
    elseif contains ~= nil and val ~= nil then
      cart = cart .. "*2> Error in template: " .. lab .. 
                     " has both fields CONTAINS and VALUE\n"
      return false
    end

    if t == nil then -- field does not exist in table being checked
      if not optional then        -- must have a value or a contains
        cart = cart .. "*3* Error: " .. lab .. " missing (not optional)\n"
        return false
      elseif default == nil then  -- can be void
        return true
      else                        -- check the optional value
        table.setvar(torig, field, default)
        t = default
        precart = "*-> Error in DEFAULT for " .. lab .. " in template:\n"
      end
    end

    allstrict = allstrict or treestr

    if val ~= nil then     -- a leave in tree
      local ty = type(val)
      if ty ~= type(t) then  -- checks type
        ty = ty .. ((ty == "table") and " {...}" or "")
        cart = cart .. "*4* Error: " .. lab ..
                       " incorrect type; must be a " .. ty .. "\n"
        return false
      end

      local ok1, msg = test(t)  -- check with TEST function
      if not ok1 then
        cart = cart .. "*5* Error: " .. lab .. " invalid value. " .. msg
        return false
      end
    else                       -- a branch in tree
      if type(t) ~= "table" then
        cart = cart .. "*6* Error: " .. lab .. " must be a table"
        return false
      end
      if type(contains) ~= "table" then
        cart = cart .. "*7> Error in template: " .. lab .. 
                       " field CONTAINS must be a non void table"
        return false
      end
      -- checks if value have extra fields than allowed
      if allstrict or strict == true then
        for key, _ in pairs(t) do
          local ttk = contains[key]
          if ttk == nil then -- key does not exist in template
            local name = field .. "." .. tostring(key)
            local lab = "Field <" .. label .. "." ..
                        string.gsub(name, "^%.", "") .. ">"
            cart = cart .. "*8* Error: " .. lab .. " must not exist\n"
            ok = false
          end
        end
      end
      if not ok then return ok end

      for key, value in pairs(contains) do
        local name = field .. (field == "" and "" or "." ) .. tostring(key)
        ok = maintest(value, t[key], allstrict, name)
        if not ok then return ok end
      end
    end
    precart = "" -- reset it after a DEFAULT field was analyzed
    return ok
  end

  local ok = maintest(template, t)
  return ok, cart == "" and "" or precart .. cart
end
